(function () {
    let app = angular.module('office', []);

    app.directive('imageLoaded', ['$parse', function ($parse) {
        return {
            restrict: 'A',
            link: function (scope, elem, attrs) {
                let fn = $parse(attrs.imageLoaded);
                elem.on('load', function (event) {
                    scope.$apply(function () {
                        fn(scope, {$event: event});
                    });
                });
            }
        };
    }]);

    app.directive('office', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/office.html',
            controller: function ($rootScope, $scope, $timeout) {
                Helpers.MovePropToIsolatedScope($rootScope, $scope, 'itemMove');
                Helpers.MovePropToIsolatedScope($rootScope, $scope, 'itemPreview');
                Helpers.MovePropToIsolatedScope($rootScope, $scope, 'activeBuilding');
                let ctrl = this;
                ctrl.cells = [];
                $scope.EmployeeTypes = EmployeeTypeNames;

                $scope.$on(GameEvents.WorkstationChange, () => loadWorkstationInfos());
                $scope.$on(GameEvents.GridChange, () => {
                    loadWorkstationInfos();
                    loadItems();

                    if ($rootScope.itemPreview != null && $rootScope.itemPreview.cellId != null) {
                        let cellCoordinates = GetCellCoordinates($rootScope.itemPreview.cellId, $rootScope.activeBuilding.placementOffset);
                        $rootScope.itemPreview.top = cellCoordinates.top;
                        $rootScope.itemPreview.left = cellCoordinates.left;
                        $rootScope.itemPreview.zIndex = calculateTargetZindex($rootScope.activeBuilding.width, $rootScope.itemPreview.blockedCells)

                        if($rootScope.itemPreview.item != null)
                            $rootScope.itemPreview.url = Helpers.PrepareItemPath($rootScope.itemPreview.item.imagePath, $rootScope.itemPreview.orientation);
                    }
                });

                let calculateTargetZindex = (gridSize, cellIds) => {
                    let cellId = _.max(cellIds);

                    let x = cellId % gridSize;
                    let y = Math.floor(cellId / 20) + 1;

                    return x + y;
                };

                let loadWorkstationInfos = () => {
                    ctrl.workstationInfos = $rootScope.settings.office.workstations.map(workstation => {
                        let zIndex = calculateTargetZindex($rootScope.activeBuilding.width, workstation.blockedCells);
                        let cellCoordinates = GetCellCoordinates(workstation.cellId, $rootScope.activeBuilding.placementOffset);

                        let result = {
                            workstation: workstation,
                            zIndex: zIndex,
                            menuZindex: zIndex + 1000,
                            top: cellCoordinates.top,
                            left: cellCoordinates.left,
                        };

                        if (workstation.employee != null) {
                            let employeeStat = Game.Lifecycle.EmployeeStats.find(x => x.employee.id == workstation.employee.id);
                            Object.assign(result, employeeStat);
                        }

                        return result;
                    })
                };

                let loadItems = () => {
                    ctrl.itemInfos = $rootScope.settings.office.items.map(item => {
                        let cellCoordinates = GetCellCoordinates(item.cellId, $rootScope.activeBuilding.placementOffset);
                        return {
                            item: item,
                            top: cellCoordinates.top,
                            left: cellCoordinates.left,
                            url: Helpers.PrepareItemPath(item.item.imagePath, item.orientation),
                            zIndex: calculateTargetZindex($rootScope.activeBuilding.width, item.blockedCells)
                        }
                    });
                };

                Helpers.RootScopeWatchAndDestroy($scope, 'activeBuilding', activeBuilding => {
                    if (activeBuilding == null) return;
                    ctrl.cells = [];
                    for (let i = 0; i <= ((activeBuilding.width * activeBuilding.width) - 1); i++) {
                        ctrl.cells.push({id: i, blocked: false, line: GetLineNumber(i, activeBuilding.width)});
                    }
                    initializeGrid();
                    $timeout(() => $rootScope.$broadcast(GameEvents.GridChange));
                });

                $rootScope.showPreview = function (workstation, item, width, height) {
                    $rootScope.itemPreview = {
                        workstation: workstation,
                        item: item,
                        width: width,
                        height: height,
                        orientation: 1,
                        url: item != null ? Helpers.PrepareItemPath(item.imagePath, 1) : null
                    }
                };

                $rootScope.placeWorkstation = function (workstationType) {
                    let workstation = null;
                    switch (workstationType.id) {
                        case 1:
                            workstation = {chairId: 1, computerId: 1, deskId: 1};
                            break;
                        case 2:
                            workstation = {chairId: 4, computerId: 2, deskId: 2};
                            break;
                        case 3:
                            workstation = {chairId: 4, computerId: 3, deskId: 3};
                            break;
                    }

                    workstation.tier = workstationType.tier;
                    workstation.bonus = workstationType.bonus;
                    workstation.price = workstationType.price;

                    $rootScope.showPreview(workstation, null, 2, 2);
                };

                $rootScope.sellItem = function () {
                    let target = $rootScope.itemMove;
                    let array = $rootScope.settings.office[target.type + "s"];

                    if (target.type == "workstation") {
                        if ($rootScope.itemMove.employee != null) {
                            if ($rootScope.itemMove.employee.contractId != null) {
                                let contract = $rootScope.settings.contracts.find(x => x.id == $rootScope.itemMove.employee.contractId);
                                if (contract != null) {
                                    Helpers.CancelContract(contract);
                                }
                            }
                            $rootScope.settings.unassignedEmployees.push($rootScope.itemMove.employee);
                        }
                    }

                    _.remove(array, x => x.cellId == target.cellId);

                    $rootScope.$broadcast(GameEvents.GridChange);

                    $rootScope.itemMove = null;
                };

                this.clickCell = function (cell) {
                    if ($rootScope.itemPreview != null || $rootScope.itemMove != null) {
                        if ($rootScope.itemPreview != null) {
                            this.placeItem(cell);
                        } else {
                            this.startMovingItem(cell);
                        }
                        $rootScope.$broadcast(GameEvents.WorkstationChange);
                    }

                    return false;
                };

                $rootScope.rotateSelection = () => {
                    let item = $rootScope.itemPreview || $rootScope.itemMove;

                    let getNextOrientation = function (current) {
                        let orientations = [1, 2, 3, 4];
                        let nextOrientationIndex = orientations.indexOf(current) + 1;
                        return orientations.length <= nextOrientationIndex ? orientations[0] : orientations[nextOrientationIndex];
                    };

                    item.orientation = getNextOrientation(item.orientation);
                    this.evaluateTargetLocation(item);
                    $scope.$broadcast(GameEvents.GridChange);
                };

                this.rightClickCell = function (cell) {
                    if ($rootScope.itemPreview != null) {
                        $rootScope.rotateSelection();
                    } else {
                        if ($rootScope.itemMove == null) {
                            this.startMovingItem(cell);
                        } else {
                            $rootScope.rotateSelection();
                        }
                    }
                    this.mouseoverCell(cell);
                };

                this.mouseoverCell = function (cell) {
                    let item = $rootScope.itemPreview || $rootScope.itemMove;

                    if (item == null) {
                        return false;
                    }

                    item.cellId = cell.id;
                    this.evaluateTargetLocation(item);

                    $rootScope.$broadcast(GameEvents.GridChange);
                };

                this.evaluateTargetLocation = item => {
                    let cellId = item.cellId;
                    let blockingCellIds = GetZone($rootScope.activeBuilding.width, item.width, item.height, cellId, item.orientation); // Default workstation size

                    if(blockingCellIds == null) { // Out of area
                        item.blocked = true;
                        return;
                    }

                    let blockedCellIdsInOffice = _.flatten($rootScope.settings.office.items.filter(x => !x.move).map(x => x.blockedCells))
                        .concat(_.flatten($rootScope.settings.office.workstations.filter(x => !x.move).map(x => x.blockedCells)));

                    if ( blockingCellIds != null) {
                        item.blocked = blockingCellIds.some(blockedCell => blockedCellIdsInOffice.some(x => x == blockedCell));
                        item.blockedCells = blockingCellIds;
                    }
                };

                this.startMovingItem = function (cell) {
                    if ($rootScope.itemMove == null) { // Start moving
                        let target = findHoveredTarget(cell.id);
                        if (target == null)
                            return;

                        target.move = true;
                        $rootScope.itemMove = target;

                        $rootScope.itemMove.width = target.item.width;
                        $rootScope.itemMove.height = target.item.height;
                        cell.blocked = false;
                    } else { // End move
                        if ($rootScope.itemMove.blocked) return;
                        $rootScope.itemMove.move = false;
                        $rootScope.$broadcast(GameEvents.GridChange);
                        $timeout(() => {
                            $rootScope.itemMove = null;
                        }, 0);
                    }
                };

                this.placeItem = function () {
                    if ($rootScope.itemPreview.blocked) return;

                    let price = $rootScope.itemPreview.workstation != null
                        ? $rootScope.itemPreview.workstation.price
                        : $rootScope.itemPreview.item.price;

                    let transactionLabel = $rootScope.itemPreview.workstation != null
                        ? `Tier ${$rootScope.itemPreview.workstation.tier} Workstation`
                        : Helpers.GetLocalized($rootScope.itemPreview.item.name);

                    $rootScope.safeBuy(() => {
                        if ($rootScope.itemPreview.workstation != null) {
                            let workstation = {
                                id: chance.guid(),
                                employee: null,
                                quality: 0,
                                deskId: $rootScope.itemPreview.workstation.deskId,
                                computerId: $rootScope.itemPreview.workstation.computerId,
                                chairId: $rootScope.itemPreview.workstation.chairId,
                                tier: $rootScope.itemPreview.workstation.tier,
                                upgradesApplied: [],
                                bonus: $rootScope.itemPreview.workstation.bonus,
                                cellId: $rootScope.itemPreview.cellId,
                                orientation: $rootScope.itemPreview.orientation,
                                top: $rootScope.itemPreview.top,
                                left: $rootScope.itemPreview.left,
                                item: {
                                    width: Configuration.WORKSTATION_WIDTH,
                                    height: Configuration.WORKSTATION_HEIGHT
                                },
                                blockedCells: $rootScope.itemPreview.blockedCells
                            };

                            $rootScope.settings.office.workstations.push(workstation);
                            $rootScope.$broadcast(GameEvents.WorkstationChange);
                            wallhack.sendEvent("purchased_workstation", `Tier ${ $rootScope.itemPreview.workstation.tier }`);
                        }

                        if ($rootScope.itemPreview.item != null) {
                            $rootScope.settings.office.items.push({
                                id: chance.guid(),
                                cellId: $rootScope.itemPreview.cellId,
                                blockedCells: $rootScope.itemPreview.blockedCells,
                                orientation: $rootScope.itemPreview.orientation,
                                top: $rootScope.itemPreview.top,
                                left: $rootScope.itemPreview.left,
                                item: $rootScope.itemPreview.item
                            });
                            wallhack.sendEvent("purchased_item", `${ $rootScope.itemPreview.item.name}`);
                        }


                    }, price, 'Item: ' + transactionLabel);

                    let remainingWorkstations = $rootScope.activeBuilding.workstations - $rootScope.settings.office.workstations.length;
                    if (!ShiftPressed || ($rootScope.itemPreview.workstation != null && remainingWorkstations == 0)) {
                        $rootScope.itemPreview = null;
                    } else {
                        mouseoverCell($('.cell#' + $rootScope.itemPreview.cellId));
                    }

                    $rootScope.$broadcast(GameEvents.GridChange);
                };

                ctrl.setWorkstation = function (workstation) {
                    $rootScope.view = null;
                    PlaySound(Sounds.click);
                    $rootScope.selectedWorkstation = null;
                    $timeout(() => {
                        $rootScope.selectedWorkstation = workstation;
                    });
                };
            },
            controllerAs: 'ctrl'
        };
    });


    function initializeGrid() {
        let $rootScope = GetRootScope();
        let size = ($rootScope.activeBuilding.width * 187);
        let grid = $('.grid');
        grid.css('width', size)
            .css('height', size)
            .css('margin-top', $rootScope.activeBuilding.gridOffset.top)
            .css('margin-left', $rootScope.activeBuilding.gridOffset.left);

        dragscroll.reset();
    }

    function findHoveredTarget(id) {
        let $rootScope = GetRootScope();
        let items = $rootScope.settings.office.items;
        let workstations = $rootScope.settings.office.workstations;

        let itemContainingHoveredCell = items.find(item => item.blockedCells.some(cellId => cellId == id));
        let workstationContainingHoveredCell = workstations.find(item => item.blockedCells.some(cellId => cellId == id));

        let target = itemContainingHoveredCell || workstationContainingHoveredCell;

        if (target != null) {
            target.type = (itemContainingHoveredCell != null) ? "item" : "workstation";
        }

        return target;
    }
})
();

// Used to avoid angular's ng-mouseenter to fire digest when not required
function mouseoverCell(cellObj) {
    let $rootScope = GetRootScope();
    if ($rootScope.itemPreview != null || $rootScope.itemMove != null) {
        let cellId = $(cellObj).attr('id');
        let $scope = angular.element('office .background').scope();
        let cell = $scope.ctrl.cells.find(x => x.id == cellId);
        $scope.ctrl.mouseoverCell(cell);
        if (!$rootScope.$$phase) {
            $scope.$apply();
        }
    }
}